//
//  ECBinder.h
//  ArrayObserver
//
//  Created by Mark Onyschuk on 17/01/07.
//  Copyright 2007 Extra Cayenne. All rights reserved.
//

// ECBinder is a class which provides housekeeping for views which offer Cocoa Bindings support.
// Said views implement the ECBinderClient interface, for example:
//
// - (void)binder:(ECBinder *)sender didObserveChangeInBinding:(NSString *)aBinding {
//	id aBoundStringValue = [sender valueForBinding:aBinding];
//	[self setStringValue:aBoundStringValue];
// }
//
// as well as the standard binding interface:
//
// - (void)bind:(NSString *)aBinding toObject:(id)anObject withKeyPath:(NSString *)aKeyPath options:(NSDictionary *)aDictionary {
//	[myBinder bind:aBinding toObject:anObject withKeyPath:aKeyPath options:aDictionary];
// }
// - (void)unbind:(NSString *)binding {
//	[myBinder unbind:binding];
// }
// 
// A view that uses ECBinder will typically have a single binder object manage all of it's vended bindings. For an example of such a 
// view, refer to MCMembershipView, included with ECBinder.

#import <Cocoa/Cocoa.h>

@interface ECBinder : NSObject {
	id client;
	NSMutableDictionary *bindings;
}

- (id)initWithClient:(id)aClient;

- (void)bind:(NSString *)aBinding toObject:(id)anObject withKeyPath:(NSString *)aKeyPath options:(NSDictionary *)aDictionary;
- (void)unbind:(NSString *)binding;
- (void)unbindAll;

- (BOOL)containsBinding:(NSString *)aBinding;

- (id)objectForBinding:(NSString *)aBinding;
- (id)keypathForBinding:(NSString *)aBinding;
- (id)optionsForBinding:(NSString *)aBinding;

- (id)valueForBinding:(NSString *)aBinding;
- (void)setValue:(id)aValue forBinding:(NSString *)aBinding;
- (void)setValue:(id)aValue forBinding:(NSString *)aBinding atIndex:(unsigned)anIndex;

@end

@interface NSObject (ECBinderClient)
- (void)binder:(ECBinder *)sender didObserveChangeInBinding:(NSString *)aBinding;
@end


@interface NSString (ECBinderKeypathManipulation)
- (NSString *)stringByDeletingKeypathPrefix:(NSString *)aPrefix;
@end
